namespace App\Services;

use Illuminate\Support\Facades\Http;

class QuickBooksService
{
    private $clientId;
    private $clientSecret;
    private $redirectUri;

    public function __construct()
    {
        $this->clientId = env('QUICKBOOKS_CLIENT_ID');
        $this->clientSecret = env('QUICKBOOKS_CLIENT_SECRET');
        $this->redirectUri = env('QUICKBOOKS_REDIRECT_URI');
    }

    public function getAuthorizationUrl()
    {
        $baseUrl = 'https://appcenter.intuit.com/connect/oauth2';
        $queryParams = http_build_query([
            'client_id' => $this->clientId,
            'redirect_uri' => $this->redirectUri,
            'response_type' => 'code',
            'scope' => 'com.intuit.quickbooks.accounting',
            'state' => csrf_token(),
        ]);

        return "$baseUrl?$queryParams";
    }

    public function exchangeCodeForToken($authorizationCode)
    {
        $response = Http::asForm()->post('https://oauth.platform.intuit.com/oauth2/v1/tokens/bearer', [
            'grant_type' => 'authorization_code',
            'code' => $authorizationCode,
            'redirect_uri' => $this->redirectUri,
            'client_id' => $this->clientId,
            'client_secret' => $this->clientSecret,
        ]);

        if ($response->failed()) {
            throw new \Exception('Failed to exchange authorization code for token.');
        }

        return $response->json();
    }

    public function getCompanyInfo($accessToken, $realmId)
    {
        $response = Http::withToken($accessToken)->get("https://quickbooks.api.intuit.com/v3/company/$realmId/companyinfo/$realmId");

        if ($response->failed()) {
            throw new \Exception('Failed to fetch company info.');
        }

        return $response->json();
    }

    public function fetchInventoryItems($accessToken, $realmId)
    {
        $response = Http::withToken($accessToken)->get("https://quickbooks.api.intuit.com/v3/company/$realmId/item");

        if ($response->failed()) {
            throw new \Exception('Failed to fetch inventory items.');
        }

        return $response->json();
    }

    public function updateInventoryItem($accessToken, $realmId, $itemId, $data)
    {
        $response = Http::withToken($accessToken)->post("https://quickbooks.api.intuit.com/v3/company/$realmId/item/$itemId", $data);

        if ($response->failed()) {
            throw new \Exception('Failed to update inventory item.');
        }

        return $response->json();
    }
}